/*   
 *   $Id: tuneit.c,v 1.2 2000/01/19 20:06:08 ondra Exp $
 * 
 *   Metronome and simple tool for tuning instruments
 *   Copyright (C) 1999,2000 Ondrej Palkovsky
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 * 
 *  You can contact me at 'ondrap@penguin.cz'.
*/


#include <Pilot.h>
#include <SysAll.h>
#include <SysEvtMgr.h>
#include "callback.h"

#include "tuneitRsc.h"

#define APP_ID        'OpLd'
#define PREF_ID      0
#define PREF_VERSION  2

int frequency;
int metronom;
int tone;
int sharp;
Boolean dotick=false;
Word systicks;

#define FREQ_MIN 350
#define FREQ_MAX 550

#define METRO_MIN 25
#define METRO_MAX 250

/* One half of tone */
#define TONE_KOEF 1.05946309435929530984
/* How many half tones to reach that tone, 0 position is C */
static int tone_tab[] = {-9,-7,-5,-4,-2,0,2};

struct ldPrefs {
	int frequency;
	int metronom;
	int tone; // ID of the selected button
	int sharp; // ID of the selected button
	Word systicks;
};

static Err 
StartApplication(void)
{
	struct ldPrefs prefs;
	Word size;

	size=sizeof(prefs);
	if (PrefGetAppPreferences(APP_ID,PREF_ID,&prefs,&size,0)!=PREF_VERSION) {
		prefs.frequency=440;
		prefs.metronom=60;
		prefs.tone=tdA;
		prefs.sharp=tdNatural;
		prefs.systicks=SysTicksPerSecond();
		FrmAlert(ldAlertAbout);
	}	
	frequency=prefs.frequency;
	metronom=prefs.metronom;
	systicks=prefs.systicks;
	tone=prefs.tone;
	sharp=prefs.sharp;
	
	FrmGotoForm(ldMainForm);
	return 0;
}

static Err
StopApplication(void)
{
	struct ldPrefs prefs;
	
	prefs.frequency=frequency;
	prefs.metronom=metronom;
	prefs.systicks=systicks;
	prefs.tone=tone;
	prefs.sharp=sharp;
	PrefSetAppPreferences(APP_ID,PREF_ID,PREF_VERSION,&prefs,
			      sizeof(prefs),0);
	return 0;
}


VoidPtr GetObjectPtr(Word objectID)
{
        FormPtr frm;

        frm=FrmGetActiveForm();
        return (FrmGetObjectPtr(frm,FrmGetObjectIndex(frm,objectID)));
}

static void
Click(void)
{
	SndCommandType sndcmd;
	
	sndcmd.cmd=3;
	sndcmd.param1=300;
	sndcmd.param2=50;
	sndcmd.param3=64;
	SndDoCmd(NULL,&sndcmd,0);
}

static void
SoundOn(void)
{
	SndCommandType sndcmd;
	double dest_freq;
	int ifreq;
	int count;
	int i;
	
	count = tone_tab[tone-tdC];
	dest_freq=frequency;
	
	if (sharp==tdSharp)
	  count++;
	else if (sharp==tdFlat)
	  count--;
	
	if (count>0) {
		for (i=0;i<count;i++) 
		  dest_freq*=TONE_KOEF;
	}
	else if (count<0) {
		for (i=0;i<-count;i++)
		  dest_freq/=TONE_KOEF;
	}
	
	ifreq = dest_freq;
	sndcmd.cmd=3;
	sndcmd.param1=ifreq;
	sndcmd.param2=50000;
	sndcmd.param3=64;
	SndDoCmd(NULL,&sndcmd,0);
}

static void
SoundOff(void)
{
	SndCommandType sndcmd;
	
	sndcmd.cmd=4;
	sndcmd.param1=0;
	sndcmd.param2=0;
	sndcmd.param3=0;
	SndDoCmd(NULL,&sndcmd,0);
}

static void
UpdateFreq(FormPtr frm)
{
	Word lblIndex=FrmGetObjectIndex(frm,ldFreqLbl);
	char text[6];
		
	StrPrintF(text,"%d",frequency);
	FrmHideObject(frm,lblIndex);
	FrmCopyLabel(frm,ldFreqLbl,text);
	FrmShowObject(frm,lblIndex);
/* If the tone is currently playing, change it */
	if (CtlGetValue(GetObjectPtr(ldTune)))
	  SoundOn();
}

static void
UpdateMetro(FormPtr frm)
{
	Word lblIndex=FrmGetObjectIndex(frm,ldMetroLbl);
	char text[6];
		
	StrPrintF(text,"%d",metronom);
	FrmHideObject(frm,lblIndex);
	FrmCopyLabel(frm,ldMetroLbl,text);
	FrmShowObject(frm,lblIndex);
}

static void
UpdateSysticks(FormPtr frm)
{
	Word lblIndex=FrmGetObjectIndex(frm,ldCalibLbl);
	char text[6];
		
	StrPrintF(text,"%d",systicks);
	FrmHideObject(frm,lblIndex);
	FrmCopyLabel(frm,ldCalibLbl,text);
	FrmShowObject(frm,lblIndex);
}


static Boolean 
MainFormHandleEvent(EventPtr event)
{
	Boolean    handled = false;
	FormPtr     frm;
	
#ifdef __GNUC__
   CALLBACK_PROLOGUE 
#endif
	  
	  frm=FrmGetActiveForm();
	  switch (event->eType) {
	   case keyDownEvent:
		  switch (event->data.keyDown.chr) {
		   case pageUpChr:
			  if (CtlGetValue(GetObjectPtr(ldTune))) {
				  if (frequency<FREQ_MAX) {
					  frequency++;
					  UpdateFreq(frm);
				  }
			  }
			  else if (CtlGetValue(GetObjectPtr(ldTick))) 
			    if (metronom<METRO_MAX) {
				    UpdateMetro(frm);
				    metronom++;
			    }
			  
			  handled=true;
			  break;
		   case pageDownChr:
			  if (CtlGetValue(GetObjectPtr(ldTune))) {
				  if (frequency>FREQ_MIN) {
					  frequency--;
					  UpdateFreq(frm);
				  }
			  }
			  else if (CtlGetValue(GetObjectPtr(ldTick)))
			    if (metronom>METRO_MIN) {
				    UpdateMetro(frm);
				    metronom--;
			    }
			  
			  handled=true;
			  break;
		  }
		  break;
	   case frmOpenEvent:	
		  FrmDrawForm(frm);

		  UpdateFreq(frm);
		  UpdateMetro(frm);
		  UpdateSysticks(frm);
		  CtlSetValue(GetObjectPtr(tone),true);
		  CtlSetValue(GetObjectPtr(sharp),true);
		  
		  handled=true;
		  break;
	   case ctlSelectEvent:
		  switch (event->data.ctlSelect.controlID) {
		   case tdC:
		   case tdD:
		   case tdE:
		   case tdF:
		   case tdG:
		   case tdA:
		   case tdH:
			  tone = event->data.ctlSelect.controlID;
			  UpdateFreq(frm);
			  break;
		   case tdSharp:
		   case tdFlat:
		   case tdNatural:
			  sharp = event->data.ctlSelect.controlID;
			  UpdateFreq(frm);
			  break;
		   case ldHelpBut:
			  FrmHelp(ldHelp2);
			  break;
		   case ldAboutBut:
			  FrmAlert(ldAlertAbout);
			  break;
		   case ldTune:
			  if (CtlGetValue(GetObjectPtr(ldTune))) {
				  SoundOn();				  
				  if (CtlGetValue(GetObjectPtr(ldTick))) {
					  CtlSetValue(GetObjectPtr(ldTick),0);
					  dotick=false;
				  }
			  }
			  else
			    SoundOff();
			  handled=true;
			  break;
		   case ldTick:
			  if (CtlGetValue(GetObjectPtr(ldTick))) {
				  dotick=true;
				  if (CtlGetValue(GetObjectPtr(ldTune))) {
					  CtlSetValue(GetObjectPtr(ldTune),0);
					  SoundOff();
				  }
			  }
			  else
			    dotick=false;
			  handled=true;
			  break;
		  }
		  break;
	   case ctlRepeatEvent:
		  switch(event->data.ctlRepeat.controlID) {
		   case ldFreqUp:
			  if (frequency<FREQ_MAX) {
				  frequency++;
				  UpdateFreq(frm);
			  }
			  break;
		   case ldFreqDown:
			  if (frequency>FREQ_MIN) {
				  frequency--;
				  UpdateFreq(frm);
			  }
			  break;
		   case ldMetroUp:
			  if (metronom<METRO_MAX) {
				  metronom++;
				  UpdateMetro(frm);
			  }
			  break;
		   case ldMetroDown:
			  if (metronom>METRO_MIN) {
				  metronom--;
				  UpdateMetro(frm);
			  }
			  break;
		   case ldCalibUp:
			  systicks++;
			  UpdateSysticks(frm);
			  break;
		   case ldCalibDown:
			  systicks--;
			  UpdateSysticks(frm);
			  break;
		  }
	  }	
	
#ifdef __GNUC__
	CALLBACK_EPILOGUE 
#endif
	  return handled;
}
	  

static Boolean 
ApplicationHandleEvent(EventPtr event)
{
	FormPtr  frm;
	Int    formId;
	Boolean  handled = false;
	
	if (event->eType == frmLoadEvent) {
		//Load the form resource specified in the event then activate it
		formId = event->data.frmLoad.formID;
		frm = FrmInitForm(formId);
		FrmSetActiveForm(frm);
		// Set the event handler for the form.  The handler of the currently
		// active form is called by FrmDispatchEvent each time it is called
		switch (formId) {		
		 case ldMainForm:
			FrmSetEventHandler(frm,MainFormHandleEvent);
			break;
		}
		handled = true;	       
	}
	
	return handled;
}

static void
EventLoop(void)
{
	EventType  event;
	Word      error;
	Long timeout;
	
	do {
		if (dotick)
		  timeout=(systicks*60)/metronom;
		else
		  timeout=evtWaitForever;
		
		EvtGetEvent(&event,timeout);
		
		if (event.eType==nilEvent) {
/* Don't turn off if metronom in use */
			EvtResetAutoOffTimer();
			Click();
			continue;
		}
		
		if (! SysHandleEvent(&event))
		  if (! MenuHandleEvent(0, &event, &error))
		    if (! ApplicationHandleEvent(&event))
		      FrmDispatchEvent(&event);
	} while (event.eType != appStopEvent);
}

DWord
PilotMain(Word launchCode, Ptr cmdPBP, Word launchFlags)
{
	Err err=0;
	
	if (launchCode == sysAppLaunchCmdNormalLaunch) {		
		if ((err = StartApplication()) == 0) {
			EventLoop();
			FrmCloseAllForms();
			StopApplication();
		}
	}
	return 0;
}
